 /**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: ConnectorAPI.cpp $
   Last Modification:
    $Author: Robert $
    $Modtime: $
    $Revision: 7138 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: no
     WinCE        : no
 
   Description:
    Implementation of the netX Connector API
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     3         20.07.2015  RM       - Changed registry path in CreateRegistryKey() from
                                      HKEY_LOCAL_MACHINE to HKEY_CURRENT_USER
                                    - Added information string for MessageBox on registry error
     2         26.11.09    RM       review
     1         04.06.09    RM       created 

 
**************************************************************************************/

/*****************************************************************************/
/*! \file RS232Connector/ConnectorAPI.cpp                                    */
/*   netX Connector API implementation                                       */
/*****************************************************************************/

#include "stdafx.h"

#include "RS232Layer.h"
#include "ConnectorAPI.h"
#include "netXConnectorErrors.h"
#include "RS232Config.h"
#include "RS232ConfigDlg.h"

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_RS232 netX RS232 Connector                    */
/*! \{                                                                       */
/*****************************************************************************/

///////////////////////////////////////////////////////////////////////////////////////////
/// UUID of RS232 Connector
///////////////////////////////////////////////////////////////////////////////////////////
const UUID g_tConnectorUUID = { 0x4df3ea2f, 0xc989,  0x46bd, {0x81, 0xbf, 0xa0, 0x2e, 0xc6, 0xf8, 0x21, 0x7d} };

/* This is the single instance RS232 connector object */
CRS232Layer* g_pcRS232Layer = NULL;

/* Global configuration object for the TCP layer, TRUE loads config from registry */
CRS232Config  g_cRS232Config(TRUE);

/***************************************************************************
* 
*   
*   Connector API Functions
*
*
*
*
*
******************************************************************************/

/*****************************************************************************/
/*! Returns identifier and UUID of the connector
*   \param  szIdentifier Returns identifier of the connector
*   \param  pvUUID       Returns UUID of the connector
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConGetIdentifier( char* szIdentifier, void* pvUUID)
{
  if(!szIdentifier)
    return NXCON_DRV_INVALID_POINTER;

  if(!pvUUID)
    return NXCON_DRV_INVALID_POINTER;
  
  /* Copy identifier of the connector */
  strcpy( szIdentifier, RS232_CONNECTOR_IDENTIFIER);

  /* Copy UUID of the connector */
  UUID* ptUUID = (UUID*)pvUUID;
  *ptUUID      = g_tConnectorUUID;
  
  return NXCON_NO_ERROR;
}

/******************************************************************************/
/*! Open the connector
*   \param  pfnDevNotifyCallback  Function pointer for the callback
*   \param  pvUser                User pointer
*   \return NXCON_NO_ERROR on success                                         */
/******************************************************************************/
long APIENTRY netXConOpen( PFN_NETXCON_DEVICE_NOTIFY_CALLBACK pfnDevNotifyCallback, void* pvUser)
{
  if(g_pcRS232Layer)
    return NXCON_DRV_WAS_OPENED_BEFORE;

  g_pcRS232Layer = new CRS232Layer;

  return g_pcRS232Layer->Init( pfnDevNotifyCallback, pvUser);
}

/*****************************************************************************/
/*! Close an open connector 
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConClose(void)
{ 
  long lRet = NXCON_DRV_NOT_INITIALIZED;

  if( g_pcRS232Layer)
  {
    g_pcRS232Layer->Deinit();

    delete g_pcRS232Layer;
    g_pcRS232Layer = NULL;
    
    lRet = NXCON_NO_ERROR;
  }
  return lRet;
}

/*****************************************************************************/
/*! Create device interface for the connector
*   \param  szDeviceName    Name of the interface (e.g. "\\.\COM1")
*   \return PCONNECTOR_INTERFACE on success                                  */
/*****************************************************************************/
PCONNECTOR_INTERFACE APIENTRY netXConCreateInterface( const char* szDeviceName)
{ 
  if(!g_pcRS232Layer)
    return NULL;

  return g_pcRS232Layer->GetInterface( szDeviceName);
}

/*****************************************************************************/
/*! Start the connector interface
*   \param  pvInterface     Pointer of the interface 
*   \param  pfnReceiveData  Function pointer of the receive callback
*   \param  pvUser          User pointer
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConStartInterface( PCONNECTOR_INTERFACE pvInterface, PFN_NETXCON_DEVICE_RECEIVE_CALLBACK pfnReceiveData, void* pvUser)
{ 
  if(!g_pcRS232Layer)
    return NXCON_DRV_NOT_INITIALIZED;

  if(!pvInterface)
    return NXCON_DRV_INVALID_POINTER;

  long lRet = NXCON_NO_ERROR;

  EnterCriticalSection(&g_pcRS232Layer->m_tcsInterfaceLock);  

  if(NXCON_NO_ERROR == (lRet = g_pcRS232Layer->ValidateInterface(pvInterface)))
  {
    lRet = reinterpret_cast<CRS232Interface*>(pvInterface)->Start( pfnReceiveData, pvUser);
  }
  
  LeaveCriticalSection(&g_pcRS232Layer->m_tcsInterfaceLock);  

  return lRet;
}

/*****************************************************************************/
/*! Stop the connector interface 
*   \param  pvInterface   Pointer of the interface 
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConStopInterface(PCONNECTOR_INTERFACE pvInterface)
{ 
  if(!g_pcRS232Layer)
    return NXCON_DRV_NOT_INITIALIZED;

  if(!pvInterface)
    return NXCON_DRV_INVALID_POINTER;

  long lRet = NXCON_NO_ERROR;

  EnterCriticalSection(&g_pcRS232Layer->m_tcsInterfaceLock);  

  if(NXCON_NO_ERROR == (lRet = g_pcRS232Layer->ValidateInterface(pvInterface)))
  {
    reinterpret_cast<CRS232Interface*>(pvInterface)->Stop();
  }
  
  LeaveCriticalSection(&g_pcRS232Layer->m_tcsInterfaceLock);

  return lRet;
}

/*****************************************************************************/
/* Send data about the connector interface 
*   \param  pvInterface    Pointer of the interface 
*   \param  pabData        Data pointer 
*   \param  ulDataLen      Length of the send data 
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConSendInterface( PCONNECTOR_INTERFACE pvInterface, unsigned char* pabData, unsigned long ulDataLen)
{ 
  if(!g_pcRS232Layer)
    return NXCON_DRV_NOT_INITIALIZED;

  if( !pvInterface  ||
      !pabData        )
    return NXCON_DRV_INVALID_POINTER;

  long lRet = NXCON_NO_ERROR;

  EnterCriticalSection(&g_pcRS232Layer->m_tcsInterfaceLock);  

  if(NXCON_NO_ERROR == (lRet = g_pcRS232Layer->ValidateInterface( pvInterface)))
  {
    lRet = reinterpret_cast<CRS232Interface*>(pvInterface)->Send( pabData, ulDataLen);
  }

  LeaveCriticalSection(&g_pcRS232Layer->m_tcsInterfaceLock);

  return lRet;
}

/*****************************************************************************/
/* Get the configuration
*   \param  eCmd         eCMD_CONFIG_GETLEN/eCMD_CONFIG_GETSTRING/
*                        eCMD_DIALOG_GETLEN/eCMD_DIALOG_GETSTRING
*   \param  pvConfig     Pointer to result of the request
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConGetConfig( NETX_CONNECTOR_CONFIG_CMD_E eCmd, void* pvConfig)
{ 
  long lRet = NXCON_NO_ERROR;

  if( !pvConfig)
    return NXCON_DRV_INVALID_POINTER;

  g_cRS232Config.Lock();
  CString csConfig; 
  lRet = g_cRS232Config.CreateString( csConfig);
  g_cRS232Config.Unlock();

  if (NXCON_NO_ERROR != lRet)
    return lRet;

  switch (eCmd)
  {
    case eCMD_CONFIG_GETLEN:
      {
        unsigned long* pulLength = (unsigned long*)pvConfig;
        *pulLength = csConfig.GetLength();
      }
      break;

    case eCMD_CONFIG_GETSTRING:
      {
        char* pszString = (char*)pvConfig;
        strcpy(pszString, csConfig.GetBuffer());
      }
      break;
    
    case eCMD_DIALOG_GETLEN:
      {
        CRS232ConfigDlg* pcDialog = CRS232ConfigDlg::GetInstance();    

        if (pcDialog->GetWindowHandle())
        {
          /* Get configuration string from config object */
          CString csConfig;
          if( NXCON_NO_ERROR == (lRet = pcDialog->GetConfiguration( csConfig)))
          {
            unsigned long* pulLength = (unsigned long*)pvConfig;
            *pulLength = csConfig.GetLength();
          }
        } else
        {
          lRet = NXCON_DRV_NOT_INITIALIZED;
        }
      }
      break;

    case eCMD_DIALOG_GETSTRING:
      {
        CRS232ConfigDlg* pcDialog = CRS232ConfigDlg::GetInstance();    

        if (pcDialog->GetWindowHandle())
        {
          /* Get configuration string from config object */
          CString csConfig;
          if( NXCON_NO_ERROR == (lRet = pcDialog->GetConfiguration( csConfig)))
          {
            char* pszString = (char*)pvConfig;
            strcpy(pszString, csConfig.GetBuffer());
          }
        } else
        {
          lRet = NXCON_DRV_NOT_INITIALIZED;
        }
      }
      break;

    default:
      lRet = NXCON_DRV_INVALID_PARAMETER;
      break;
  }

  return lRet;
}

/*****************************************************************************/
/*! Set the connector configuration
*   \param  eCmd         eCMD_CONFIG_SETSTRING/eCMD_DIALOG_SETSTRING
*   \param  szConfig     Configuration string
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConSetConfig( NETX_CONNECTOR_CONFIG_CMD_E eCmd, const char* szConfig)
{ 
  long lRet = NXCON_NO_ERROR;

  if( !szConfig)
    return NXCON_DRV_INVALID_POINTER;
  
  switch (eCmd)
  {
    case eCMD_CONFIG_SETSTRING:
      g_cRS232Config.Lock();
      if (NXCON_NO_ERROR == (lRet = g_cRS232Config.ParseString( szConfig)))
      {
        /* Store Configuration to registry */
        CString szRegPath;
        long lRet = NXCON_NO_ERROR;
        if( NXCON_NO_ERROR != (lRet = g_cRS232Config.StoreToRegistry(szRegPath)))
        {
          CString csText;
          csText.Format(_T("Failed to store RS232 connector configuration\r<%s>"), szRegPath);
          MessageBox( NULL, csText, _T("netXTransport DLL connector error"), MB_ICONEXCLAMATION);
        }
      }
      g_cRS232Config.Unlock();
      
      /* Reinit Connector */
      if(g_pcRS232Layer)
        g_pcRS232Layer->Reinit();
      break;

    case eCMD_DIALOG_SETSTRING:
      {
        CRS232ConfigDlg* pcDialog = CRS232ConfigDlg::GetInstance();    

        if (pcDialog->GetWindowHandle())
        {
          /* Set configuration string from config object */
          lRet = pcDialog->SetConfiguration( szConfig);
        } else
        {
          lRet = NXCON_DRV_NOT_INITIALIZED;
        }
      }
      break;

    default:
      lRet = NXCON_DRV_INVALID_PARAMETER;
      break;
  }

  return lRet;
}

/*****************************************************************************/
/*! Get connector information
*   \param  eCmd            eSTATE/eUUID/eTYPE/eESTABLISH_TIMEOUT
*   \param  ulSize          Size of information structure
*   \param  pvConnectorInfo Connector information structure
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConGetInformation( NETX_CONNECTOR_INFO_E eCmd, unsigned long ulSize, void* pvConnectorInfo)
{
  if (!pvConnectorInfo)
    return NXCON_DRV_INVALID_POINTER;

  long lRet = NXCON_NO_ERROR;

  switch(eCmd)
  {
    case eSTATE :
      if(sizeof(NETX_CONNECTOR_INFO_STATE_U) <= ulSize)
      {
    
        NETX_CONNECTOR_INFO_STATE_U* ptConnectorState = (NETX_CONNECTOR_INFO_STATE_U*)pvConnectorInfo;
    
        /* Populate Connector Info structure */
        g_cRS232Config.Lock();
        ptConnectorState->tBfState.fEnable = g_cRS232Config.IsConnectorEnabled();
        g_cRS232Config.Unlock();
      }else
      {
        lRet = NXCON_DRV_BUFFER_TOO_SHORT;
      }
    break ;

    case eUUID:
      if(sizeof(UUID) <= ulSize)
      {
        UUID* ptUUID = (UUID*)pvConnectorInfo;
        *ptUUID = g_tConnectorUUID;
      } else
      {
        lRet = NXCON_DRV_BUFFER_TOO_SHORT;
      }
    break;

    case eTYPE:  
      if(sizeof(NETX_CONNECTOR_TYPE_U) <= ulSize)
      {
        NETX_CONNECTOR_TYPE_U* ptConnectorType = (NETX_CONNECTOR_TYPE_U*)pvConnectorInfo;
        
        /* Clear Connector Type structure */
        ptConnectorType->ulType = 0;
        /* Populate Connector Type structure */
        ptConnectorType->tBfType.fNXDRV_TYPE_SERIAL = TRUE;
        ptConnectorType->tBfType.fNXDRV_TYPE_USB    = TRUE;

      }else
      {
        lRet = NXCON_DRV_BUFFER_TOO_SHORT;
      }
    break;

    case eIDENTIFIER:
      if(strlen(RS232_CONNECTOR_IDENTIFIER) < ulSize)
      {
        /* Copy identifier of the connector */
        strcpy( (char*)pvConnectorInfo, RS232_CONNECTOR_IDENTIFIER);        
      } else
      {
        lRet = NXCON_DRV_BUFFER_TOO_SHORT;
      }
    break;

    case eSHORT_INTERFACE_NAME:
      {
        INTERFACE_NAME_TRANSLATION_T* ptTranslate = (INTERFACE_NAME_TRANSLATION_T*)pvConnectorInfo;

        /* RS232 has same short and long name, so we can use the source directly */
        
        std::string szSource( ptTranslate->szSource);
        size_t posSeparator  = szSource.find("_");

        if ( string::npos == posSeparator)
        {
          /* Interface name does not match the pattern */
          lRet = NXCON_DRV_INVALID_PARAMETER;

        } else
        {

          /* Split szSource into interface name and board name */
          std::string szInterfaceName = szSource.substr( 0, posSeparator);
          std::string szBoardName     = szSource.substr( posSeparator + 1, string::npos - posSeparator);

          /* Validity check */
          if(0 != szInterfaceName.compare(0, strlen(RS232_CONNECTOR_IDENTIFIER), RS232_CONNECTOR_IDENTIFIER))
          {
            /* This is not our long interface format */
            lRet = NXCON_CONF_INVALID_INTERFACE;

          } else
          {
            if(NULL == g_pcRS232Layer->GetInterface(szInterfaceName))
            {
              /* TODO: If we don't find the long name, we need to create it, so ODM can access unconfigured 
                      devices with a direct path */
              lRet = NXCON_CONF_INVALID_INTERFACE;
            } else
            {
              /* Assemble short interface name (e.g. COM1) */
              _snprintf(ptTranslate->szDestination, ptTranslate->ulDestinationSize, "%s_%s", szInterfaceName.c_str(), szBoardName.c_str());
              lRet = NXCON_NO_ERROR;
            }
          }
        }
      }
      break;

    case eLONG_INTERFACE_NAME:
      {
        INTERFACE_NAME_TRANSLATION_T* ptTranslate = (INTERFACE_NAME_TRANSLATION_T*)pvConnectorInfo;

        /* Serial connector has same short and long filename */
        if(strlen(ptTranslate->szSource) < ptTranslate->ulDestinationSize)
        {
          strncpy(ptTranslate->szDestination, ptTranslate->szSource, ptTranslate->ulDestinationSize);
          lRet = NXCON_NO_ERROR;
        } else
        {
          lRet = NXCON_DRV_BUFFER_TOO_SHORT;
        }
      }
      break;
    
    case eDESCRIPTION:
      if(strlen(RS232_CONNECTOR_DESCRIPTION) < ulSize)
      {
        /* Copy identifier of the connector */
        strcpy( (char*)pvConnectorInfo, RS232_CONNECTOR_DESCRIPTION);        
      } else
      {
        lRet = NXCON_DRV_BUFFER_TOO_SHORT;
      }
    break;

    default:
      lRet = NXCON_DRV_INVALID_PARAMETER;
      break;
  }
  
  return lRet;
}

/*****************************************************************************/
/*! Get interface information
*   \param  pvInterface     Interface pointer
*   \param  eCmd            eSTATE/eINTERFACE_NAME/eHUMAN_READABLE_NAME/eUUID
*   \param  ulSize          Size of information structure
*   \param  pvInterfaceInfo Interface information structure
*   \return NXCON_NO_ERROR on success                                        */
/*****************************************************************************/
long APIENTRY netXConGetInformationInterface( PCONNECTOR_INTERFACE pvInterface, NETX_INTERFACE_INFO_E eCmd, unsigned long ulSize, void* pvInterfaceInfo)
{
  if (!pvInterfaceInfo)
    return NXCON_DRV_INVALID_POINTER;

  if(!pvInterface)
    return NXCON_DRV_INVALID_POINTER;

  if(!g_pcRS232Layer)
    return NXCON_DRV_INVALID_POINTER;

  long lRet = NXCON_NO_ERROR;
  
  /* Get interface information */
  if ( NXCON_NO_ERROR == (lRet = g_pcRS232Layer->ValidateInterface(pvInterface)))
  {
    CRS232Interface* pcIF = reinterpret_cast<CRS232Interface*>(pvInterface);
    switch(eCmd)
    {
      case eINTERFACE_STATE:   
        if(sizeof(unsigned long) <= ulSize)
        {
          unsigned long* pulInterfaceState = (unsigned long*)pvInterfaceInfo;
          *pulInterfaceState = pcIF->GetCtrlState();

        }else
        {
          lRet = NXCON_DRV_BUFFER_TOO_SHORT;
        }
      break;
     
      case eSEND_TIMEOUT:  
        if(sizeof(unsigned long) <= ulSize)
        {
          unsigned long* pulInterfaceTimeout = (unsigned long*)pvInterfaceInfo;
          *pulInterfaceTimeout = pcIF->GetSendTimeout();

          
        }else
        {
          lRet = NXCON_DRV_BUFFER_TOO_SHORT;
        }
      break;

      case eRESET_TIMEOUT:  
        if(sizeof(unsigned long) <= ulSize)
        {
          unsigned long* pulInterfaceTimeout = (unsigned long*)pvInterfaceInfo;
          *pulInterfaceTimeout = pcIF->GetResetTimeout();

        }else
        {
          lRet = NXCON_DRV_BUFFER_TOO_SHORT;
        }
      break;

      case eKEEPALIVE_TIMEOUT:  
        if(sizeof(unsigned long) <= ulSize)
        {
          unsigned long* pulInterfaceTimeout = (unsigned long*)pvInterfaceInfo;
          *pulInterfaceTimeout = pcIF->GetKeepAliveTimeout();

        }else
        {
          lRet = NXCON_DRV_BUFFER_TOO_SHORT;
        }
      break;

      default:
        lRet = NXCON_DRV_INVALID_PARAMETER;
        break;
    }
  }

  return lRet;
}

/*****************************************************************************/
/*! Display configuration dialog
*   \param  hParentWnd Parent Window
*   \param  szConfig   Configuration string to configure dialog
*   \return Handle to dialog window                                          */
/*****************************************************************************/
HWND APIENTRY netXConCreateDialog (HWND hParentWnd, const char* szConfig)
{
  if( !hParentWnd)
    return NULL;

  CRS232ConfigDlg*  pcDialog = CRS232ConfigDlg::GetInstance();
  HWND              hWnd     = pcDialog->GetWindowHandle();

  if (!hWnd)
  {
    if (szConfig)
    {
      /* Create dialog and use the configuration supplied via config string */
      hWnd = pcDialog->Create( hParentWnd, szConfig);
    
    } else
    {
      /* No config string supplied. Create dialog without initial config */
      hWnd = pcDialog->Create( hParentWnd);
    }
  }

  return hWnd;
}

/*****************************************************************************/
/*! Remove configuration dialog                                              */
/*****************************************************************************/
void APIENTRY netXConEndDialog ()
{
  CRS232ConfigDlg* pcDialog = CRS232ConfigDlg::GetInstance();    
  
  if (pcDialog)
  {
    pcDialog->Destroy();
  }
}

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
